

#include "ADC.h"


/**
 *******************************************************************************
 * @brief       Enable/Disable ADC block
 * @param[in]   NewState: config control bit
 *              This parameter can be: DRV_ENABLE or DRV_DISABLE.
 * @return      none
 *******************************************************************************
 */
void ADC_Cmd ( FunctionalState NewState )
{
    if ( NewState == DRV_ENABLE )
        ADCON0 |= ADCEN;
    else
        ADCON0 &= ~ADCEN;

}

/**
 *******************************************************************************
 * @brief       Input Channel Selection for ADC analog multiplexer.
 * @param[in]   ADCSel: config conversion channel select
 *  @arg\b      ADC_AIN0
 *  @arg\b      ADC_AIN1
 *  @arg\b      ADC_AIN2
 *  @arg\b      ADC_AIN3
 *  @arg\b      ADC_AIN4
 *  @arg\b      ADC_AIN5
 *  @arg\b      ADC_AIN6
 *  @arg\b      ADC_AIN7
 *  @arg\b      ADC_IntVREF
 * @return      none
 *******************************************************************************
 */
void ADC_Channel_Select ( ADC_ChannelDef ADCSel )
{
    U8 orginSFRPI;;
    orginSFRPI = SFRPI;
    ADCON0 &= ~ ( CHS3 | CHS2 | CHS1 | CHS0 );
    ADCON0 |= ADCSel & ( CHS3 | CHS2 | CHS1 | CHS0 );
    SFRPI = 0;
    ADCFG0 &= ~ACHS;
    ADCFG0 |= ADCSel & ACHS;
    SFRPI = orginSFRPI;
}

/*
 *******************************************************************************
 * @brief       conversion once
 * @param[in]   none
 * @return      none
 *******************************************************************************
 */
void ADC_StartConversion()
{
    ADCON0 |= ADCS;
}

/*
 *******************************************************************************
 * @brief       get conversion complete flag
 * @param[in]   none
 * @return      none
 *******************************************************************************
 */
FlagReturn ADC_GetConversionFlag()
{
    if ( ADCON0 & ADCI )
        return DRV_Set;

    return DRV_Clr;
}

/*
 *******************************************************************************
 * @brief       clear conversion complete flag
 * @param[in]   none
 * @return      none
 *******************************************************************************
 */
void ADC_ClearFlag()
{
    ADCON0 &= ~ADCI;
}


/**
 *******************************************************************************
 * @brief       select ADC conversion clock
 * @param[in]   ADCSel: config conversion channel select
 *  @arg\b      ADC_SYSCLK
 *  @arg\b      ADC_SYSCLKD2
 *  @arg\b      ADC_SYSCLKD4
 *  @arg\b      ADC_SYSCLKD8
 *  @arg\b      ADC_SYSCLKD16
 *  @arg\b      ADC_SYSCLKD32
 *  @arg\b      ADC_S0TOFD2
 *  @arg\b      ADC_T2OFD2
 * @return      none
 *******************************************************************************
 */
void ADC_Clock_Select ( ADC_ClockDef ADCClkSel )
{
    SFRPI = 0x00;
    ADCFG0 &= ~ ( ADCKS2 | ADCKS1 | ADCKS0 );
    ADCFG0 |= ADCClkSel & ( ADCKS2 | ADCKS1 | ADCKS0 );
}

/**
 *******************************************************************************
 * @brief       config ADC conversion data Alignment mode
 * @param[in]   AlignMode: data alignment mode select (Right/Left)
 *  @arg\b      ADC_LeftJustified : Preserved old date
 *  @arg\b      ADC_RightJustified : Overwritten by new data
 * @return      none
 *******************************************************************************
 */
void ADC_DataAlignment_Select ( ADC_DataAlignModeDef AlignMode )
{
    SFRPI = 0x00;
    ADCFG0 &= ~ ( ADRJ );
    ADCFG0 |= AlignMode;
}

/**
 *******************************************************************************
 * @brief       config ADC conversion data with offset setting
 * @param[in]   PowerSel: select convertsion power level
 *  @arg\b      ADC_HighestPower,
 *  @arg\b      ADC_MiddleHigh,
 *  @arg\b      ADC_MiddleLow,
 *  @arg\b      ADC_LowPower,
 * @return      none
 *******************************************************************************
 */
void ADC_SetConversionPower ( ADC_ConversionPowerDef PowerSel )
{
    U8 orginSFRPI;

    orginSFRPI = SFRPI;
    SFRPI = 0x03;
    ADCFG3 &= ~ ( ADPS1 | ADPS0 );
    ADCFG3 |= PowerSel;
    SFRPI = orginSFRPI;
}



/**
 *******************************************************************************
 * @brief       get ADC conversion data
 * @param[in]   none
 * @return      return unsigned int 16bit ADC conversion data
 *******************************************************************************
 */
U16 ADC_GetConversionData()
{
    WTYPE RetDat;

    RetDat.B[0] = ADCDH;
    RetDat.B[1] = ADCDL;

    return RetDat.W;
}


/**
 *******************************************************************************
 * @brief       Initial ADC
 * @param[in]   none
 * @return      none
 *******************************************************************************
 */
void Initial_ADC()
{
    ADC_Clock_Select ( ADC_SYSCLKD2 );
    ADC_DataAlignment_Select ( ADC_RightJustified );
    ADC_Channel_Select ( ADC_AIN3 );
    ADC_SetConversionPower(ADC_HighestPower);
    ADC_Cmd ( DRV_ENABLE );
}



